<?php
/*
  These actions are useful in documenting APIs.

  Class action will create a link to the class' page or output it in plain text if current
  page is the one describing that class.
  Current class is determined by taking level 1 heading's first word (before space, if any).
  Character case matters.

    {{Class ClassName}} or {{Class Class1, Class2, ...}}

  Func and Prop actions will create 2 or 3 links (multiple fields are different, see below):
  1. ClassName - only link if current doc isn't already desciring ClassName;
  2. -> or ::  - links to the section on instance/static functions/properties of ClassName;
  3. field     - links to that field (a funcion or a property).

    {{Func ClassName->Method}}            - or ...Method()
    {{Func ClassName::StaticMethod}}      - or ...StaticMethod()
    {{Prop ClassName->property}}          - or ...$property
    {{Prop ClassName::staticProperty}}    - or ...$staticProperty
    {{Func ->Method}} or {{Prop ::prop}}  - for current class
    {{Func ClassName}} or {{Prop ClassName}}  - without field they work as {{Class ClassName}}

  Multiple funcs/props allowed: {{Func Class->Method, Class::Static, ...}}
  You can put spaces around -> or ::, e.g. {{Prop Class -> prop}}

    If multiple funcs/props are specified class and scope (:: or ->) will only be output
    if they differ from previous field. Example:
      {{Func Class->Method, Class->Method2, Class::Method3, Class2::Method}}
    outputs "Class->Method, Method2, ::Method3, Class2::Method". As a shortcut you could write this:
      {{Func Class->Method, Method2, ::Method3, Class2::Method}}

  To save keystrokes you can create aliases:

    {{FieldAlias alias => real}}            - adds alias (as short as 1 char) for all 3 field types:
                                            {{Class alias}} works as {{Class real}}
                                            {{Func alias->Method}} or {{Func ClassName->alias}}
    {{FieldAlias class alias => real}}     - 1 specific alias: "class", "func" or "prop" (case-insensitive):
                                            {{Class alias}} works as {{Class real}}
                                            {{Prop alias::prop}} works but not {{Prop ClassName::alias}}
    You can also use "=" instead of "=>'.
    ALiasing affects all class/func/prop calls even if they come earlier in the doc.
    Spaces around "=>" or "=" are optional: {{FieldAlias real=alias, real=>alias}}
    Real-life example:    {{FieldAlias class Tpl=MyTemplater}}

  CONFIGURATION - fields added to UWikiSettings:

    * $FieldAlias - predefined aliases: array( 'class|func|prop' => array('alias' => 'realName', ...) )
      Defaults to $rootURL/Classes/
    * $ClassesRoot - root URL of dir with class' pages (relative to $rootURL even if absolute).
    * $ClassFieldAnchor - pattern of anchor attached to each function/property
      on the target doc page; "$" symbol is replaced by the field name; defaults to "$".
*/

class Ufieldalias_Root extends UWikiBaseAction {
  public $runsImmediately = true;

  function Execute($format, $params) {
    foreach ($params as $leftSide => $real) {
      @list($group, $alias) = explode(' ', $leftSide, 2);
      if (!isset($alias)) {
        $groups = array('class', 'func', 'prop');
        $alias = $leftSide;
      } else {
        $groups = array( strtolower($group) );
      }

      $real = ltrim($real, '>');
      foreach ($groups as $group) { self::NewAlias($this->settings, $group, $alias, $real); }
    }
  }

  static function NewAlias($settings, $group, $alias, $real) {
    $alias = trim($alias);
    $real = trim($real);
    if (!self::IsEmptyStr($real) and !self::IsEmptyStr($alias)) {
      $settings->format->settings->FieldAlias[ trim($group) ][$alias] = $real;
    }
  }

  static function ResolveAlias($settings, $group, $name) {
    $real = &$settings->format->settings->FieldAlias[$group][$name];
    return isset($real) ? $real : $name;
  }
}

class Uclass_Root extends UWikiBaseAction {
  public $htmlTag = 'kbd';
  public $htmlClasses = array('class-link');

  function Execute($format, $params) {
    foreach ($params as $param => $unused) {
      $children = $this->Add($param);
      if ($children) {
        $groupObj = $this->children[] = $this->NewElement('Uclass_One');
        $groupObj->children = $children;
        $this->children[] = $this->NewElement('Uclass_Separator');
      }
    }

    array_pop($this->children);  // last link shouldn't be followed by separator.
    $format->appendMe = !empty($this->children);
  }

    function Add($className) {
      $classObj = $this->NewElement('Uclass_ClassName');
      $classObj->Set($className);
      return array($classObj);
    }

  // Lower priority so e.g. == %%title%% == will be parsed before we run.
  function Priority($format, $params) { return (int) UWikiMaxPriority * 0.3; }
}

  class Uclass_Separator extends UWikiStringMessage {
    public $message = '{{class: separator';
    public $defaultMessage = ', ';
  }

  class Uclass_One extends UWikiBaseElement {
    public $htmlTag = 'span';
    public $htmlClasses = array('one');
  }

  class Uclass_Kbd extends Upre_Root {
    public $isBlock = false;
    public $htmlTag = 'kbd';
  }

    class Uclass_Base extends Uclass_Kbd {
      public $isBlock = false;
      public $htmlTag = 'kbd';

      function CurrentClass() {
        $doc = $this->settings->format ? $this->settings->format->origDoc : $this->doc;
        return strtok($doc->TextTitle(), ' ');
      }

      function PageOf($className) {
        $root = &$this->settings->ClassesRoot;
        $root = self::IsEmptyStr($root) ? 'Classes/' : trim($root, '\\/').'/';
        return $this->settings->rootURL.$root. $className.$this->settings->linkExt;
      }
    }

      class Uclass_ClassName extends Uclass_Base {
        public $htmlClasses = array('class');
        public $href;

        function Set($className) {
            self::IsEmptyStr($className) and $className = $this->CurrentClass();
            $className = Ufieldalias_Root::ResolveAlias($this->settings, 'class', $className);

          $this->SetRaw($className);
          if ($this->CurrentClass() !== $className) {
            $this->htmlTag = 'a';
            $this->htmlAttributes['href'] = $this->PageOf($className);
          }
        }
      }

class Ufunc_Root extends Uclass_Root {
  public $htmlClasses = array('func-link');

  public $fieldType = 'methods';
  public $aliasGroup = 'func';
  public $fieldNamePrefix = '';
  public $fieldNameSuffix = '()';

  public $prevClass, $prevStatic;

  function Add($param) {
    // Class::Func or ::Func or Func.
    @list($className, $field) = explode('->', $param, 2);
    $isStatic = !isset($field);
    $isStatic and @list($className, $field) = explode('::', $param, 2);

    if ($this->prevClass) {
      $className === '' and $className = $this->prevClass;

      if (!isset($field)) {
        $field = $className;
        $className = $this->prevClass;
        $isStatic = $this->prevStatic;
      }
    }

    if (!isset($field)) {
      return parent::Add($param);
    } else {
      $className = trim($className);
      $field = trim($field, "\x00..\x20".$this->fieldNamePrefix.$this->fieldNameSuffix);
      $field = Ufieldalias_Root::ResolveAlias($this->settings, $this->aliasGroup, $field);

      $children = array();

        if ($className !== $this->prevClass) {
          $children[] = $classObj = $this->NewElement('Uclass_ClassName');
          $classObj->Set($className);
        }

        if ($className !== $this->prevClass or $isStatic !== $this->prevStatic) {
          $children[] = $scopeObj = $this->NewElement('Ufunc_Scope');
          $anchor = ($isStatic ? 'static' : 'instance').'_'.$this->fieldType;
          $scopeObj->Set($className, $anchor, $isStatic);
        }

        $children[] = $prefix = $this->NewElement('Uclass_Kbd');
        $prefix->SetRaw($this->fieldNamePrefix);

          $children[] = $nameObj = $this->NewElement('Ufunc_FieldName');
          $nameObj->Set($className, $field, $isStatic);
          $nameObj->SetRaw($field);

        $children[] = $suffix = $this->NewElement('Uclass_Kbd');
        $suffix->SetRaw($this->fieldNameSuffix);

      $this->prevClass = $className;
      $this->prevStatic = $isStatic;

      return $children;
    }
  }
}

  class Ufunc_Scope extends Uclass_Base {
    public $htmlTag = 'a';
    public $htmlClasses = array('scope-');

    function Set($className, $anchor, $isStatic) {
      if ($this->CurrentClass() === $className) {
        $page = $this->settings->pageForSelfAnchorLinks;
      } else {
        $page = $this->PageOf($className);
      }

      $this->htmlAttributes['href'] = "$page#".mb_strtolower($anchor);
      $this->SetRaw( $isStatic ? '::' : '->' );
      $this->htmlClasses[0] .= $isStatic ? 'static' : 'instance';
    }
  }

    class Ufunc_FieldName extends Ufunc_Scope {
      public $htmlClasses = array('field-');

      function Set($className, $anchor, $isStatic) {
        $pattern = &$this->settings->ClassFieldAnchor;
        $pattern or $pattern = '$';
        $anchor = str_replace('$', $anchor, $pattern);

        parent::Set($className, $anchor, $isStatic);
      }
    }

class Uprop_Root extends Ufunc_Root {
  public $htmlClasses = array('prop-link');

  public $fieldType = 'properties';
  public $aliasGroup = 'prop';
  public $fieldNamePrefix = '$';
  public $fieldNameSuffix = '';
}
